/**
* \file: automounterctl.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>

#include "automounterctl.h"
#include "automounter_api.h"
#include "utils/version_info.h"
#include "handle_coldplug_devs_cmd/handle_coldplug_devs_cmd.h"
#include "handle_mnt_umnt_request_cmd/handle_umount_request_cmd.h"
#include "handle_mnt_umnt_request_cmd/handle_remount_request_cmd.h"
#include "handle_monitor_cmd/handle_monitor_cmd.h"

static automounterctl_command_vtable_t *automounterctl_cmd_list[]=
{
		&handle_coldplug_devs_cmd_vtable,
		&handle_umount_request_cmd_vtable,
		&handle_remount_request_cmd_vtable,
		&handle_monitor_cmd_vtable,
		NULL
};

static automounterctl_command_vtable_t *current_command=NULL;

static error_code_t automounterctl_parse_command(int argc,char *argv[]);
error_code_t automounterctl_find_command_handler(const char *command);
static void automounterctl_printusage(void);
static void automounterctl_printhelp(void);
static void automounterctl_printversion(void);


//----------------------------------------------- device sync api members ----------------------------------------------
error_code_t automounterctl_init(int argc, char *argv[])
{
	error_code_t result;
	current_command=NULL;

	result=automounterctl_parse_command(argc,argv);
	if (result==RESULT_OK)
		if (current_command->init!=NULL)
			result=current_command->init();

	if (result==RESULT_OK)
		if (current_command->parse_args!=NULL)
			result=current_command->parse_args(argc,argv);

	return result;
}

error_code_t automounterctl_start(void)
{
	if (current_command->start!=NULL)
		return current_command->start();
	return RESULT_OK;
}

void automounterctl_deinit(void)
{  
	if (current_command!=NULL)
		if (current_command->deinit!=NULL)
			current_command->deinit();
}

void automounterctl_error_message(error_code_t err)
{
	if (err==RESULT_NORESOURCE)
		fprintf(stderr, "Error: Out of memory.");
}
//-------------------------------------------------------------------------------------------------------------------------

//------------------------------------------- private members -------------------------------------------------------------
static error_code_t automounterctl_parse_command(int argc,char *argv[])
{
	error_code_t result=RESULT_OK;

	if (argc<2)
	{
		printf("\nAutomounter needs a command passed as command line option.\n");
		automounterctl_printusage();
		result=RESULT_INVALID_ARGS;
	}

	if (result==RESULT_OK)
	{
		if (strcmp(argv[1],"--help")==0 || strcmp(argv[1],"-h")==0 )
		{
			automounterctl_printhelp();
			result=RESULT_HELP_PRINTED;
		}

		if (strcmp(argv[1],"--version")==0 || strcmp(argv[1],"-v")==0 )
		{
			automounterctl_printversion();
			result=RESULT_HELP_PRINTED;
		}
	}

	if (result==RESULT_OK)
		result=automounterctl_find_command_handler(argv[1]);

	return result;
}

static void automounterctl_printhelp(void)
{
	int itr;
	printf("\n");
	printf("Automounter Control Utility - Used to control the ADIT automounter daemon.\n\n");
	printf("automounterctl [<COMMAND>] [OPTIONS]\n");
	printf("\t-h,--help:\t\tdisplays some help and exits.\n");
	printf("\t-v,--version:\tdisplays version information and exits.\n\n");
	printf("Supported commands:\n");
	for (itr=0;automounterctl_cmd_list[itr]!=NULL;itr++)
		printf("\t%20s: \t%s\n",automounterctl_cmd_list[itr]->command, automounterctl_cmd_list[itr]->command_description);
	printf("\nUse automounterctl <COMMAND> --help to get more information on \nthe usage of the respective command.\n\n");
}

static void automounterctl_printusage(void)
{
	printf("\n");
	printf("automounterctl [<COMMAND>] [OPTIONS]\n");
	printf("\t-h,--help:	displays some help and exits.\n");
	printf("\t-v,--version:\tdisplays version information and exits.\n\n");
}

static void automounterctl_printversion(void)
{
	printf("automounterctl:\t\t%s\n", VERSION_INFO_FORMATED_LINE);
	printf("libautomounter.so:\t%s\n\n", automounter_api_get_version_string());
}

error_code_t automounterctl_find_command_handler(const char *command)
{
	int itr;

	current_command=NULL;

	for(itr=0;automounterctl_cmd_list[itr]!=NULL;itr++)
	{
		if (strcmp(automounterctl_cmd_list[itr]->command,command)==0)
		{
			current_command=automounterctl_cmd_list[itr];
			break;
		}
	}

	if (current_command==NULL)
	{
		printf("Invalid command: '%s'.\n",command);
		automounterctl_printusage();
		return RESULT_INVALID_ARGS;
	}
	else
		return RESULT_OK;
}
//-------------------------------------------------------------------------------------------------------------------------
